/////////////////////////////////////////////////////////////////////////////////
//
// nuvectormath_gen.inl
//
/////////////////////////////////////////////////////////////////////////////////

template<bool HW> 
inline NuVector3Base<HW> NuVector3Base<HW>::LerpS( const NuVector3Base<HW>& v1, const NuVector3Base<HW>& v2, const NuScalarBase<HW>& s )
{
	return v1 + ( (v2 - v1) * s );
}

template<bool HW> 
inline NuVector3Base<HW> NuVector3Base<HW>::Lerp( const NuVector3Base<HW>& v1, const NuVector3Base<HW>& v2, const float t )
{
	return LerpS( v1, v2, NuScalarBase<HW>( t ) );
}

template<bool HW> 
inline bool NuVector3Base<HW>::Overlap( const NuVector3Base<HW>& minA, const NuVector3Base<HW>& maxA, const NuVector3Base<HW>& minB, const NuVector3Base<HW>& maxB )
{
	NuVector3Base<HW> test1 = ( maxB >= minA );
	NuVector3Base<HW> test2 = ( maxA >= minB );
	test1 = MinPerElem( test1, test2 );
	return test1.AllTrue( );
}

template<bool HW> 
inline bool NuVector3Base<HW>::Contains( const NuVector3Base<HW>& minA, const NuVector3Base<HW>& maxA, const NuVector3Base<HW>& minB, const NuVector3Base<HW>& maxB )
{
	NuVector3Base<HW> test1 = ( maxB < maxA );
	NuVector3Base<HW> test2 = ( minB > minA );
	test1 = MinPerElem( test1, test2 );
	return test1.AllTrue( );
}

template<bool HW> 
inline NuVector4Base<HW> NuVector4Base<HW>::LerpS( const NuVector4Base<HW>& v1, const NuVector4Base<HW>& v2, const NuScalarBase<HW>& s )
{
	return v1 + ( (v2 - v1) * s );
}

template<bool HW> 
inline NuVector4Base<HW> NuVector4Base<HW>::Lerp( const NuVector4Base<HW>& v1, const NuVector4Base<HW>& v2, const float t )
{
	return LerpS( v1, v2, NuScalarBase<HW>( t ) );
}

template<bool HW> 
inline NuMatrixBase<HW> NuMatrixBase<HW>::Ortho( ) const
{	
	NuVector3Base<HW> x = GetX( ).Norm( );
	NuVector3Base<HW> y = GetY( );
	NuVector3Base<HW> z = GetZ( );
	NuVector3Base<HW> t = GetT( );
	z = x.Cross( y ).Norm( );
	y = z.Cross( x );
	return NuMatrixBase<HW>( x, y, z, t );
}

template<bool HW> 
inline NuQuatBase<HW>::NuQuatBase( const NuMatrixBase<HW>& m )
{
	float res[4];
	float tr = m.Get(0,0) + m.Get(1,1) + m.Get(2,2);
	if ( tr > 0.0f ) // check the diagonal
	{
		float s = NuFsqrt( tr + 1.0f );
		res[3] = s / 2.0f;
		s = 0.5f / s;
		res[0] = (m.Get(1,2) - m.Get(2,1)) * s;
		res[1] = (m.Get(2,0) - m.Get(0,2)) * s;
		res[2] = (m.Get(0,1) - m.Get(1,0)) * s;
	} 
	else // diagonal is negative
	{		 
		int i = 0;
		if (m.Get(1,1) > m.Get(0,0)) i = 1;
		if (m.Get(2,2) > m.Get(i,i)) i = 2;

		int nxt[3] = { 1, 2, 0 };
		int j = nxt[i];
		int k = nxt[j];

		float s = NuFsqrt( (m.Get(i,i) - (m.Get(j,j) + m.Get(k,k))) + 1.0f );
		res[i] = s * 0.5f;
		if ( s != 0.0f ) s = 0.5f / s;
		res[3] = (m.Get(j,k) - m.Get(k,j)) * s;
		res[j] = (m.Get(i,j) + m.Get(j,i)) * s;
		res[k] = (m.Get(i,k) + m.Get(k,i)) * s;
	}
	
	SetX( res[0] );
	SetY( res[1] );
	SetZ( res[2] );
	SetW( res[3] );
}

template<bool HW> 
inline NuQuatBase<HW> NuQuatBase<HW>::BlendS( const NuQuatBase<HW>& q1, const NuQuatBase<HW>& q2, NuScalarBase<HW>& s1, NuScalarBase<HW>& s2 )
{
	return q1 * s1 + q2 * s2;
}

template<bool HW> 
inline NuQuatBase<HW> NuQuatBase<HW>::Blend( const NuQuatBase<HW>& q1, const NuQuatBase<HW>& q2, float f1, float f2 )
{
	NuScalarBase<HW> s1 = NuScalarBase<HW>( f1 );
	NuScalarBase<HW> s2 = NuScalarBase<HW>( f2 );
	return BlendS( q1, q2, s1, s2 );
}

template<bool HW> 
inline NuQuatBase<HW> NuQuatBase<HW>::LerpS( const NuQuatBase<HW>& q1, const NuQuatBase<HW>& q2, NuScalarBase<HW>& t )
{
	return q1 + ( (q2 - q1) * t );
}

template<bool HW> 
inline NuQuatBase<HW> NuQuatBase<HW>::Lerp( const NuQuatBase<HW>& q1, const NuQuatBase<HW>& q2, float t )
{
	NuScalarBase<HW> st = NuScalarBase<HW>( t );
	return LerpS( q1, q2, st );
}

template<bool HW> 
inline NuQuatBase<HW> NuQuatBase<HW>::FastSlerp( const NuQuatBase<HW>& q1, const NuQuatBase<HW>& q2, float t ) 
{ 	
	NuQuatBase<HW> q2temp;
	float  dot, even, odd, theta, recipSqrt;
	float  a, b, sina, sinb;

	dot = q1.Dot( q2 );
	if ( dot < 0.0f ) // negate to shorter path
	{
		dot = -dot;
		q2temp = -q2;		
	}
	else
	{
		q2temp = q2;
	}

	// a delta of 0.85f (Cos 32 degrees) gives a good switching point from linear to spherical interpolation
	const float delta = 0.85f;
	if ( dot > delta ) // do lerp
		return Lerp( q1, q2temp, t ).Norm( );

	dot  = min( dot, 0.995f );
	even = 2.218480716f + .2234036692f * dot * dot;
	odd  = 2.441884385f * dot;

	theta = NuFsqrt( even - odd ) - NuFsqrt( even + odd ) + 1.570796327f + .6391287330f * dot;
	recipSqrt = 1.0f / NuFsqrt( 1 - dot * dot );

	a = ( 1.0f - t ) * theta - 1.570796327f;
	a = a * a;
	b = t * theta - 1.570796327f;
	b = b * b;

	sina = ((( 0.00002315401401f * a - 0.001385370794f) * a + 0.04166358517f) * a - 0.4999990537f) * a + 0.9999999535f;
	sinb = ((( 0.00002315401401f * b - 0.001385370794f) * b + 0.04166358517f) * b - 0.4999990537f) * b + 0.9999999535f;

	a = sina * recipSqrt;
	b = sinb * recipSqrt;

	return Blend( q1, q2temp, a, b ); 
}

template<bool HW> 
inline NuMatrixBase<HW> NuMatrixBase<HW>::GetRotFromTo( const NuVector3Base<HW>& from, const NuVector3Base<HW>& to )
{
	NuVector3Base<HW> axis = from.Cross( to );
		
	float sinTheta = axis.Length( );
	float theta = NuASinf( sinTheta );
	if ( sinTheta != 0.0f )
		axis /= sinTheta;

	return GetRotAxis( axis, theta );		
}

template<bool HW> 
inline void NuScalarBase<HW>::Serialize( NuArchive& archive )
{
	float x = *this;
	archive.Serialize( x );
	*this = x;
}

template<bool HW> 
inline void NuVector3Base<HW>::Serialize( NuArchive& archive )
{
	float x = GetX( );
	float y = GetY( );
	float z = GetZ( );
	archive.Serialize( x );
	archive.Serialize( y );
	archive.Serialize( z );
	SetX( x );
	SetY( y );
	SetZ( z );
}

template<bool HW> 
inline void NuVector4Base<HW>::Serialize( NuArchive& archive )
{
	float x = GetX( );
	float y = GetY( );
	float z = GetZ( );
	float w = GetW( );
	archive.Serialize( x );
	archive.Serialize( y );
	archive.Serialize( z );
	SetX( x );
	SetY( y );
	SetZ( z );
	SetW( w );
}

template<bool HW> 
inline void NuQuatBase<HW>::Serialize( NuArchive& archive )
{
	float x = GetX( );
	float y = GetY( );
	float z = GetZ( );
	float w = GetW( );
	archive.Serialize( x );
	archive.Serialize( y );
	archive.Serialize( z );
	SetX( x );
	SetY( y );
	SetZ( z );
	SetW( w );
}

#ifdef NUVM_VUMATH_SUPPORT
template<bool HW> 
inline NuMatrixBase<HW>::NuMatrixBase( const NUMTX& m )
{
	float* f = (float*)&m;
	LoadAsFloat16( f );
}
#endif

template<bool HW> 
inline void NuMatrixBase<HW>::Serialize( NuArchive& archive )
{
	for ( int i = 0; i < 4; i++ )
	{
		float x = Get( i, 0 );
		float y = Get( i, 1 );
		float z = Get( i, 2 );
		float w = Get( i, 3 );
		archive.Serialize( x );
		archive.Serialize( y );
		archive.Serialize( z );
		Set( i, 0, x );
		Set( i, 1, y );
		Set( i, 2, z );
		Set( i, 3, w );
	}
}
